/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    19 Sep 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          19 Sep 2022     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#define MAIN_GLOBALS

/* Includes -------------------------------------------------------------------*/
#include <stdio.h>
#include "main.h"

volatile uint32_t    intrMMC;

/** @addtogroup Projects_Examples_MD
  * @{
  */

/* Private constants ----------------------------------------------------------*/
/* Private types --------------------------------------------------------------*/
md_rcu_init_typedef rcu_initStruct =    /**< RCU init structure */
{
    MD_RCU_MPRE_MCO_DIV_1,
    MD_RCU_MCO_DISABLE,
    MD_RCU_HRCDIV_DIV_1,
    MD_RCU_PPRE_HCLK_DIV_1,
    MD_RCU_HPRE_SYSCLK_DIV_1,
    MD_RCU_SYSCLK_HRC,
    (RCU_CON_HRCON_MSK ),
};

md_uart_init_typedef uart_initStruct =    /**< UART init structure */
{
  MD_UART_BAUDRATE_115200,
  MD_UART_LCON_LSB_FIRST,
  MD_UART_LCON_PS_EVEN,
  MD_UART_LCON_STOP_1,
  MD_UART_LCON_DLS_8,
};

/* Private variables ----------------------------------------------------------*/
/* Private macros -------------------------------------------------------------*/
/* Public functions -----------------------------------------------------------*/
/* Public function prototypes ------------------------------------------------*/

/* Private function prototypes ------------------------------------------------*/
void Iomux(void);
void SysPeriInit(void);
void ACMInit(uint32_t);
/**
  * @brief  main.
  * @param  None
  * @retval None
  */
int main(void)
{
  __disable_irq();
  md_rcu_init_set(RCU, &rcu_initStruct);
  SysPeriInit();
  NVIC_EnableIRQ(MMC_IRQn);      //MMC interrupt  
  md_uart_init(UART1, &uart_initStruct);
  __enable_irq();
  intrMMC = 0;
  printf("Low Voltage Detector(LVD) Sample\r\n");
  ACMInit(MMC_ADC_CTRL0_SEL_VCM_VDDA_DIV2);
  md_mmc_enable_it_lvdo_chg(MMC);

  md_mmc_enable_en_vpump(MMC);                             
  md_mmc_set_sel_vpump(MMC, MD_MMC_SEL_VPUMP_4V2);         // 0: 3.8V, 1: 4.0V, 2: 4.2V, 3: 4.4V
  md_mmc_enable_en_vdda_float(MMC);                              // 0: 3.6V, 1: 3.2V, 2: 2.8V, 3: 2.4V
  md_mmc_enable_en_vrefo(MMC);                             
  md_mmc_set_sel_gref(MMC, MD_MMC_SEL_GREF_0V5_VDDA);      // 0: floating, 1: 0.36V, 2: 1.08, 3: 1.8V
  md_mmc_enable_en_vrgen(MMC);
  
  md_mmc_enable_hys_lvd(MMC);    
  md_mmc_set_lvd_mode(MMC, MD_MMC_LVD_LVD_MODE_12OF271_VDD);
  md_mmc_set_sel_pin(MMC,MD_MMC_LVD_SEL_PIN_1V2);
//  md_mmc_set_lvd_ctrl_sel_nin(MMC,MD_MMC_LVD_SEL_NIN_07OF16_VDD);
  md_mmc_enable_en_lvd(MMC);  
  
  md_mmc_enable_en_mmc(MMC);

  while(1);  
}

/**
  * @brief  Peripheral Init
  * @note   Enable peripheral clock
  * @param  None
  * @retval None
  */
void SysPeriInit(void)
{
  md_rcu_enable_gpiod(RCU);
  md_rcu_enable_gpioc(RCU);
  md_rcu_disable_hdiven(RCU);

  md_rcu_disable_suart1en(RCU);
  md_rcu_disable_iwdten(RCU);
  md_rcu_disable_wwdt(RCU);
  md_rcu_disable_bs16t1(RCU);
  md_rcu_disable_gp32c4t1(RCU);

  md_rcu_disable_gp16c2t2(RCU);
  md_rcu_disable_gp16c2t1(RCU);
  md_rcu_enable_uart1(RCU);
  md_rcu_disable_spi1(RCU);
  md_rcu_enable_mmcen(RCU);
  md_rcu_disable_lcden(RCU);
    
  md_rcu_set_ipcfg_vpumpsw_hrc(RCU);
  md_rcu_set_ipcfg_vpumpdiv(RCU,4);
  md_rcu_enable_ipcfg_vpumpen(RCU);
    
  md_rcu_set_ipcfg_mmcsw_hosc(RCU);
  md_rcu_set_ipcfg_mmcdiv(RCU,0);  
  md_rcu_enable_ipcfg_mmcen(RCU);
  
  Iomux();
}

/**
  * @brief  Configure UART1 I/O Multiplexer
  * @note   PC4: UART1_TX.
  *         PC3: UART1_RX (Internal weak pull-up).
  * @param  None
  * @retval None
  */
void Iomux(void)
{
  md_gpio_set_pin_pull(GPIOC, MD_GPIO_PIN_3,   MD_GPIO_PULL_UP);
  md_gpio_set_pin_function0_7(GPIOC, MD_GPIO_PIN_3,   MD_GPIO_AF0);
  md_gpio_set_pin_function0_7(GPIOC, MD_GPIO_PIN_4,   MD_GPIO_AF0);
  md_gpio_set_pin_mode(GPIOC, MD_GPIO_PIN_3,   MD_GPIO_MODE_FUNCTION);
  md_gpio_set_pin_mode(GPIOC, MD_GPIO_PIN_4,   MD_GPIO_MODE_FUNCTION);

  md_gpio_set_pin_pull(GPIOD, MD_GPIO_PIN_0,   MD_GPIO_PULL_UP);
  md_gpio_set_pin_function0_7(GPIOD, MD_GPIO_PIN_0,   MD_GPIO_AF4);
  md_gpio_set_pin_mode(GPIOD, MD_GPIO_PIN_0,   MD_GPIO_MODE_FUNCTION);

}

/**
  * @brief  ACM Initial function
  * @param  select: Analog-to-Digital Converter Common-Mode Voltage Selection
            @arg MMC_ADC_CTRL0_SEL_VCM_VDDA_DIV2
            @arg MMC_ADC_CTRL0_SEL_VCM_1_2V
  * @retval none
  */
void ACMInit(uint32_t select)
{
#define CHIP_ID_ADDR 0x0898
#define QFN32_CHIP_ID 0x099102C0
    
    uint32_t chip_id;
    
    md_fc_read_info(CHIP_ID_ADDR, &chip_id);
    printf("chipid: 0x%x\r\n", chip_id);
    if(chip_id == QFN32_CHIP_ID)
        SYSCFG->CFG |= 1<<8;  
    
    md_mmc_set_sel_vcm(MMC, select);
    md_mmc_enable_ad_acmbuf(MMC);
}

/**
  * @brief  Uart sendchar.
  * @param  arg: char to be sent.
  * @retval data to be sent.
  */
uint8_t  sendchar(uint8_t ch)
{
  while (UART1->STAT & (UART_STAT_TFFULL_MSK));    /* Tx FIFO full */

  UART1->TXBUF = ch;            /* Sent byte */
  return (ch);
}

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/


